/***********************************************************************************************************************
* File Name		: MMS101_SDK_for_Arduino_Sample_Sketch.ino
* Version		: 1.0.0.1
* Device(s)		: Arduino UNO/DUE
* Description	: MMS101 SDK sample program
***********************************************************************************************************************/
/**********************************************************************************************************************
* History : DD.MM.YYYY Version	Description
*		  : 03.06.2024 1.0.0.0	First Release
*         : 22.09.2025 1.0.0.1  UNO_R4対応
***********************************************************************************************************************/
//==============================================================================
// INCLUDES
//==============================================================================
#include <SPI.h>

//==============================================================================
// Sample Sketch Version
//==============================================================================
uint8_t sketch_version[4] = {'1','0','0','1'};		// Release version

//==============================================================================
// DEFINES AND MACROS for Arduino Settings
//==============================================================================
//#define DEBUG						// for Debug

#define UNO				1
#define DUE				2
#define LEONARDO		3
#define MEGA2560		4
#define UNO_R4          5
#define ARDUINO_VER   UNO            // Used Arduino Board

#if(ARDUINO_VER == UNO)
	#define CSB 7
#elif(ARDUINO_VER == DUE)
	#define CSB 7
#elif(ARDUINO_VER == LEONARDO)
	#define CSB 7
#elif(ARDUINO_VER == MEGA2560)
	#define CSB 7
#elif(ARDUINO_VER == UNO_R4)
    #define CSB 7
#else
	#define CSB 7
#endif
//==============================================================================
// Setting Prameter
//==============================================================================
// Data Rate Setting[usec]
#define DATA_RATE	10000

// Temperature Correction[cycle]
// 0: No Temperature Correction
// N: Temperature correction is performed every N times
#define TEMP_CYCLE	0

// Log Separater
char sep[2] = ",";

//==============================================================================
// DEFINES AND MACROS
//==============================================================================
// command set for SPI1 Conversion Board
#define SPI1_CMD_START		0xF0			// START Command
#define SPI1_CMD_DATA		0xE0			// DATA Command
#define SPI1_CMD_DATA2		0xE2			// DATA2 Command
#define SPI1_CMD_RESTART	0xC0			// RESTART Command
#define SPI1_CMD_BOOT		0xB0			// BOOT Command
#define SPI1_CMD_STOP		0xB2			// STOP Command
#define SPI1_CMD_RESET		0xB4			// RESET Command
#define SPI1_CMD_STATUS		0x80			// STATUS Command
#define SPI1_CMD_VERSION	0xA2			// VERSION Command
#define SPI1_CMD_COEFF_FX	0x30			// COEFF_FX Command
#define SPI1_CMD_COEFF_FY	0x32			// COEFF_FY Command
#define SPI1_CMD_COEFF_FZ	0x34			// COEFF_FZ Command
#define SPI1_CMD_COEFF_MX	0x36			// COEFF_MX Command
#define SPI1_CMD_COEFF_MY	0x38			// COEFF_MY Command
#define SPI1_CMD_COEFF_MZ	0x3A			// COEFF_MZ Command
#define SPI1_CMD_INTERVAL	0x44			// INTERVAL Command

// command set from Arduino IDE (Serial Monitor/Serial Plotter)
#define IDE_CMD_START			'S'			// Measurement start
#define IDE_CMD_START2			's'			// Measurement start
#define IDE_CMD_STOP			'P'			// Measurement stop
#define IDE_CMD_STOP2			'p'			// Measurement stop

// command set from Evaluation Program(PC Application)
#define APP_CMD_INST				'T'			// Instruction code
#define APP_CMD_VERSION				0x15		// Read firmware version command from application
#define APP_CMD_START				0x23		// Start command from application
#define APP_CMD_STOP				0x33		// Stop command from application
#define APP_CMD_MEASURE_TIME		0x43		// Interval measurement command from application
#define APP_CMD_RESTART_CYCLES		0x44		// Interval restart command from application

#define ERROR_CODE_NO_ERROR			0x00		// No Error
#define ERROR_CODE_NOT_SUPPORT		0x10		//

// Status Code
#define STC_OK			0x00				// No Error
#define STC_BUSY		0x01				// BUSY
#define STC_ERR1		0x81				// Nut Support
#define STC_ERR2		0x82				// Illegal Command
#define STC_ERR3		0x83				// Illegal Parameter

// State ID
#define STT_INITIAL	  0x00
#define STT_STANDBY	  0x01
#define STT_BOOT	  0x02
#define STT_READY	  0x03
#define STT_MEASURE	  0x04
#define STT_RESET	  0x05
#define STT_ERROR	  0xFF

//==============================================================================
// VARIABLES
//==============================================================================
int32_t mms101_adc[6];					// mms101 adc data
int32_t mms101_data[6];					// mms101 matrix data
int32_t mms101_coeff[6][6];				// mms101 matrix operation correction coefficients

uint8_t  errorCode;						// Status Code
uint16_t measureStatus;					// Measure Status
uint8_t  stateId;						// State ID

boolean measureFlag = false;			// 
boolean outputFlag = false;				// true="P"or"p", false="S"or"s" 
boolean standAloneFlag = true;			// true="Serial monitor", false="Evaluation Progmram"

uint32_t startTimerCount;
uint32_t nowTimerCount;
uint32_t pastTime;
uint32_t waitTime;
uint32_t intervalCycle;

String axisLabel[6] = {"FX","FY","FZ","MX","MY","MZ"};

//=============================================================================
// setup
//==============================================================================
void setup() {
	uint8_t axis;
	uint8_t count;

    Serial.begin(250000);                        // serial boudrate: 921600bps
    
#if(ARDUINO_VER == UNO_R4)
    Serial.dtr();
#endif
   
	// Initialize port
	pinMode(CSB, OUTPUT);
	digitalWrite(CSB, HIGH);

	// Initialize SPI
	SPI.begin();
    SPI.beginTransaction(SPISettings(2000000, MSBFIRST, SPI_MODE3));
	//dummy
	SPI.transfer(0x00);

	// Initialize parameter
	errorCode = STC_OK;
	measureStatus = 0x0000;
	stateId = STT_INITIAL;
	
	// wait 20ms
	delay(20);

	//----------
	// RESET
	//----------
	errorCode = sensor_reset();
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.print("Reset Error!:");
		Serial.println(errorCode);
#endif
		return;
	}
	// wait 20ms
	delay(20);

	//----------
	// STATUS
	//----------
	errorCode = sensor_status();
#ifdef DEBUG
	Serial.print("MeasureStatus:");
	Serial.println(measureStatus,HEX);
	Serial.print("StateId:");
	Serial.println(stateId,HEX);
#endif
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.print("Reset Error!:");
		Serial.println(errorCode);
#endif
		return;
	}
	// wait 20ms
	delay(20);

	//----------
	// BOOT
	//----------
	errorCode = sensor_boot();
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.print("Boot Error!:");
		Serial.println(errorCode);
#endif
		return;
	}

	//----------
	// wait for READY State
	//----------
	do{
		// wait 20ms
		delay(20);
		errorCode = sensor_status();
#ifdef DEBUG
		Serial.print("MeasureStatus:");
		Serial.println(measureStatus,HEX);
		Serial.print("StateId:");
		Serial.println(stateId,HEX);
#endif
	}while(stateId < STT_READY);

	//----------
	// COEFF Fx,Fy,Fz,Mx,My,MZ
	//----------
	errorCode = sensor_coeff(SPI1_CMD_COEFF_FX, &mms101_coeff[0][0]);
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Coeff FX Error!");
#endif
		return;
	}
	errorCode = sensor_coeff(SPI1_CMD_COEFF_FY, &mms101_coeff[1][0]);
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Coeff FY Error!");
#endif
		return;
	}
	errorCode = sensor_coeff(SPI1_CMD_COEFF_FZ, &mms101_coeff[2][0]);
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Coeff FZ Error!");
#endif
		return;
	}
	errorCode = sensor_coeff(SPI1_CMD_COEFF_MX, &mms101_coeff[3][0]);
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Coeff MX Error!");
#endif
		return;
	}
	errorCode = sensor_coeff(SPI1_CMD_COEFF_MY, &mms101_coeff[4][0]);
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Coeff MY Error!");
#endif
		return;
	}
	errorCode = sensor_coeff(SPI1_CMD_COEFF_MZ, &mms101_coeff[5][0]);
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Coeff MZ Error!");
#endif
		return;
	}

#ifdef DEBUG
    //----------
    // Output coefficient data
    //----------
	for(axis=0;axis<6;axis++){
		Serial.print(axisLabel[axis]);
		Serial.print(",");
		for(count=0;count<6;count++){
			Serial.print(mms101_coeff[axis][count],HEX);
			Serial.print(",");
		}
		Serial.println("");
	}
#endif

	//----------
	// INTERVAL(update Temperatue Correction)
	//----------
	errorCode = sensor_interval(TEMP_CYCLE);
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Interval Error!");
#endif
		return;
	}

	//----------
	// START
	//----------
	errorCode = sensor_start();
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Start Error!");
#endif
		return;
	}

	//----------
	// start timer
	//----------
	startTimerCount = micros();
	waitTime = DATA_RATE;
	
#ifdef DEBUG
	// Clear Legend
	Serial.println(" ");

	// Legend
	Serial.print("Fx");
	Serial.print(sep);
	Serial.print("Fy");
	Serial.print(sep);
	Serial.print("Fz");
	Serial.print(sep);
	Serial.print("Mx");
	Serial.print(sep);
	Serial.print("My");
	Serial.print(sep);
	Serial.print("Mz");
	Serial.println("");
#endif

	measureFlag = true;
	outputFlag = false;
}

//==============================================================================
// loop
//==============================================================================
void loop() {
	uint8_t axis;
	uint8_t count;
	
	uint8_t command;
	uint8_t dataSize;
	uint8_t data[25];

	uint32_t temp;

	//----------
	// Error process
	//----------
	if(errorCode != STC_OK){
		return;
	}

	//----------
	// Recieve command from application
	//----------
	if(Serial.available()>1){
		command = Serial.read();			 // Read command code
		delay(1);	 // wait read(1ms)

		switch(command){
			case IDE_CMD_START:				// for serial monitor
			case IDE_CMD_START2:			// for serial monitor
				waitTime = DATA_RATE;		// set data rate
				outputFlag = true;			// START
				standAloneFlag = true;
				break;
			case IDE_CMD_STOP:				// for serial monitor
			case IDE_CMD_STOP2:				// for serial monitor
				outputFlag = false;			// STOP
				standAloneFlag = true;
				break;
			case APP_CMD_INST:      		// for application 
				dataSize = Serial.read();		// Read data length
				for(count=0; count<dataSize; count++){
					data[count] = Serial.read();	// Read command data
				}

				// command processing
				switch(data[0]){
					case APP_CMD_VERSION:
						standAloneFlag = false;
						
						data[0] = ERROR_CODE_NO_ERROR;
						data[1] = 4;
						data[2] = str2hex(sketch_version[0]);
						data[3] = str2hex(sketch_version[1]);
						data[4] = str2hex(sketch_version[2]);
						data[5] = str2hex(sketch_version[3]);
						Serial.write(data, 6);
						break;
					case APP_CMD_START:
						if(measureFlag == false){
							sensor_start();
							measureFlag = true;
						}
						outputFlag = true;
						standAloneFlag = false;

						data[0] = ERROR_CODE_NO_ERROR;
						data[1] = 0;
						Serial.write(data, 2);
						break;
					case APP_CMD_STOP:
						sensor_stop();
						measureFlag = false;
						outputFlag = false;
						standAloneFlag = false;

						data[0] = ERROR_CODE_NO_ERROR;
						data[1] = 0;
						Serial.write(data, 2);
						break;
					case APP_CMD_MEASURE_TIME:
						temp  = ((uint32_t)data[1]) << 16;
						temp += ((uint32_t)data[2]) << 8;
						temp += ((uint32_t)data[3]);
						waitTime = temp;						// interval time set
						standAloneFlag = false;

						data[0] = ERROR_CODE_NO_ERROR;
						data[1] = 0;
						Serial.write(data, 2);
						break;
					case APP_CMD_RESTART_CYCLES:
						temp  = ((uint32_t)data[1]) << 16;
						temp += ((uint32_t)data[2]) << 8;
						temp += ((uint32_t)data[3]);
						errorCode = sensor_interval(temp);		// restart interval set
						standAloneFlag = false;

						data[0] = errorCode;
						data[1] = 0;
						Serial.write(data, 2);
						break;
					default:	 // other command
						data[0] = ERROR_CODE_NOT_SUPPORT;
						data[1] = 0;
						Serial.write(data, 2);
				}
			default:
				return;
		}
	}

	//----------
	// check output request
	//----------
	if(outputFlag == false){
		return;
	}
   
	//----------
	// check wait time
	//----------
	if(sensor_wait_conversion() == false){
		return;
	}
	
	//----------
	// DATA2
	//----------
	errorCode = sensor_data2();
	if(errorCode != STC_OK){
#ifdef DEBUG
		Serial.println("Data2 Error!");
#endif
		return;
	}
	
	//----------
	// calc MATRIX
	//----------
	sensor_calc_matrix();
	if(measureStatus != 0x0000){
#ifdef DEBUG
		Serial.print("Measure Status Error!:");
		Serial.println(measureStatus, HEX);
#endif
		return;
	}

	//----------
	// output data
	//----------
	if(standAloneFlag == false){
		//----------
		// case of Evaluation Program
		//----------
		data[0] = errorCode;					// sensor_data2() errorcode
		data[1] = 23;							// data length
		data[2] = 0x80;							// fixed value
		data[3] = 0x00;							// fixed value

		// data[4]~[21]
		for(axis=0; axis<6; axis++){
			data[4 + axis*3] = mms101_data[axis] >> 16;
			data[5 + axis*3] = mms101_data[axis] >> 8;
			data[6 + axis*3] = mms101_data[axis];
		}
  
		data[22] = (uint8_t)(pastTime>>16);		// set interval time
		data[23] = (uint8_t)(pastTime>>8);
		data[24] = (uint8_t)(pastTime);
     
		Serial.write(data, 25);
	}else{
		//----------
		// case of Arduino IDE(for Serial Monitor)
		//----------
#ifdef DEBUG
		Serial.print("Data");
		Serial.print(sep);
#endif
		Serial.print(mms101_data[0]);
		Serial.print(sep);
		Serial.print(mms101_data[1]);
		Serial.print(sep);
		Serial.print(mms101_data[2]);
		Serial.print(sep);
		Serial.print(mms101_data[3]);
		Serial.print(sep);
		Serial.print(mms101_data[4]);
		Serial.print(sep);
		Serial.print(mms101_data[5]);
		Serial.println("");

#ifdef DEBUG
		Serial.print("ADC");
		Serial.print(sep);
		Serial.print(mms101_adc[0]);
		Serial.print(sep);
		Serial.print(mms101_adc[1]);
		Serial.print(sep);
		Serial.print(mms101_adc[2]);
		Serial.print(sep);
		Serial.print(mms101_adc[3]);
		Serial.print(sep);
		Serial.print(mms101_adc[4]);
		Serial.print(sep);
		Serial.print(mms101_adc[5]);
		Serial.println("");
#endif
	}
}

//==============================================================================
// START Command
//==============================================================================
uint8_t sensor_start()
{
	uint8_t ret;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_START);
	ret=SPI.transfer(0x00);
	digitalWrite(CSB, HIGH);

	return ret;
}

//==============================================================================
// RESTART Command
//==============================================================================
uint8_t sensor_restart()
{
	uint8_t ret;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_RESTART);
	ret=SPI.transfer(0x00);
	digitalWrite(CSB, HIGH);

	return ret;
}


//==============================================================================
// DATA Command
//==============================================================================
uint8_t sensor_data()
{
	uint8_t ret[21];
	uint8_t count;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_DATA);
	for(count=0;count<21;count++){
		ret[count]=SPI.transfer(0x00);
	}
	digitalWrite(CSB, HIGH);


	for(count=0;count<6;count++){
		mms101_data[count]	= (int32_t)ret[count*3+3]<<16;
		mms101_data[count] += (int32_t)ret[count*3+4]<<8;
		mms101_data[count] += (int32_t)ret[count*3+5];
		mms101_data[count] <<=8;
		mms101_data[count] >>=8;
	}

	return ret[0];
}

//==============================================================================
// DATA2 Command
//==============================================================================
uint8_t sensor_data2()
{
	uint8_t ret[21];
	uint8_t count;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_DATA2);
	for(count=0;count<21;count++){
		ret[count]=SPI.transfer(0x00);
	}
	digitalWrite(CSB, HIGH);

	measureStatus = (uint8_t)ret[1]<<8 + ret[2];

	for(count=0;count<6;count++){
		mms101_adc[count]  = (int32_t)ret[count*3+3]<<16;
		mms101_adc[count] += (int32_t)ret[count*3+4]<<8;
		mms101_adc[count] += (int32_t)ret[count*3+5];
		mms101_adc[count] <<=8;
		mms101_adc[count] >>=8;
	}

	return ret[0];
}

//==============================================================================
// BOOT Command
//==============================================================================
uint8_t sensor_boot()
{
	uint8_t ret;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_BOOT);
	ret=SPI.transfer(0x00);
	digitalWrite(CSB, HIGH);

	return ret;
}

//==============================================================================
// STOP Command
//==============================================================================
uint8_t sensor_stop()
{
	uint8_t ret;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_STOP);
	ret=SPI.transfer(0x00);
	digitalWrite(CSB, HIGH);

	return ret;
}

//==============================================================================
// RESET Command
//==============================================================================
uint8_t sensor_reset()
{
	uint8_t ret;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_RESET);
	ret=SPI.transfer(0x00);
	digitalWrite(CSB, HIGH);
	return ret;
}

//==============================================================================
// STATUS Command
//==============================================================================
uint8_t sensor_status()
{
	uint8_t ret[4];
	uint8_t count;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_STATUS);
	for(count=0;count<4;count++){
		ret[count]=SPI.transfer(0x00);
	}
	digitalWrite(CSB, HIGH);

	measureStatus = (uint8_t)ret[1]<<8 + ret[2];
	stateId = ret[3];

	return ret[0];
}

//==============================================================================
// VERSION Command
//==============================================================================
uint8_t sensor_version()
{
	uint8_t ret[7];
	uint8_t count;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_VERSION);
	for(count=0;count<7;count++){
		ret[count]=SPI.transfer(0x00);
	}
	digitalWrite(CSB, HIGH);

	return ret[0];
}

//==============================================================================
// COEFF Command
//==============================================================================
uint8_t sensor_coeff(uint8_t axis, int32_t *coeff)
{
	uint8_t ret[19];
	uint8_t count;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(axis);
	for(count=0;count<19;count++){
		ret[count]=SPI.transfer(0x00);
	}
	digitalWrite(CSB, HIGH);

	for(count=0;count<6;count++){
		coeff[count]  = (int32_t)ret[count*3+1]<<16;
		coeff[count] += (int32_t)ret[count*3+2]<<8;
		coeff[count] += (int32_t)ret[count*3+3];
		coeff[count] <<= 8;
		coeff[count] >>= 8;
	}
	return ret[0];
}

//==============================================================================
// INTERVAL Command
//==============================================================================
uint8_t sensor_interval(uint32_t interval)
{
	uint8_t ret;
	
	digitalWrite(CSB, LOW);
	SPI.transfer(SPI1_CMD_INTERVAL);
	SPI.transfer((uint8_t)(interval>>24));
	SPI.transfer((uint8_t)(interval>>16));
	SPI.transfer((uint8_t)(interval>>8 ));
	SPI.transfer((uint8_t)(interval>>0 ));
	ret=SPI.transfer(0x00);
	digitalWrite(CSB, HIGH);

	return ret;
}

//==============================================================================
// Matrix data calculation
//==============================================================================
void sensor_calc_matrix()
{
	uint8_t axis;
	uint8_t count;
	int64_t calcData;

	for(axis=0; axis<6; axis++){
		calcData = 0;
		for(count=0; count<6; count++){
			calcData += (int64_t)mms101_adc[count] * mms101_coeff[axis][count];
		}
		mms101_data[axis] = (int32_t)(calcData / 2048);
	}
}

//==============================================================================
// Waiting for conversion Function
//==============================================================================
boolean sensor_wait_conversion(void)
{
	// read now time
	nowTimerCount = micros();

	// calculate the elapsed time
	if(nowTimerCount > startTimerCount){
		pastTime = nowTimerCount - startTimerCount;
	}else{
		pastTime = 0xFFFFFFFF - startTimerCount;
		pastTime += nowTimerCount;
	}
	// check the elapsed time
	if(pastTime < waitTime){
		return false;
	}
	// update startTimerCount
	startTimerCount = nowTimerCount;
	return true;
}
//==============================================================================
// Convert String to numeral
//==============================================================================
uint8_t str2hex(uint8_t data)
{
	if(data >= '0' && data <= '9'){
		data -= '0';
	}else if(data >= 'a' && data <= 'f') {
		data -= 'a';
		data += 10;
	}else if(data >= 'A' && data <= 'F') {
		data -= 'A';
		data += 10;
	}else{
		data =0xFF;
	}
	return data;
}
