//******************************************************************************
/*!
	@file		mms101.c
	@brief		MMS101 Sample Source File

				copyright: MITSUMI Electric Co.,LTD
	@attention	none
	@warning	MITSUMI CONFIDENTIAL
*/
//******************************************************************************
/*******************************************************************************
* History : DD.MM.YYYY Version  Description
*         : 24.12.2024 1.0.0.0  First Release
*******************************************************************************/
#define __MMS101_C__

//==============================================================================
//{### MODULES USED
//==============================================================================
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <linux/spi/spidev.h>
#include <sys/time.h>
#include "spidrv.h"
#include "mms101.h"

//#define DEBUG

//==============================================================================
//{### FUNCTIONS
//==============================================================================

/*!
//==============================================================================
	@brief		Main
	@param		int argc
	@param		int **argv
	@return		int
//==============================================================================
*/
int main(int argc, char **argv)
{
	int devNo;
	int retryCount;
	int count;
	int measNumMax;
	int measureEndFlag;
	int waitTime;
	int elapsedTime;
	struct timeval tv;

	// ----------------------------------------
	// check parameter
	// ----------------------------------------
	if(argc < 3){
		mms101_usage();
		return(1);
	}
	// reset enable flag
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		mms101_data[devNo].channelEnable = 0;
	}
	// set enable flag
	for(count=1; count<argc-1; count++){
		devNo = (int)strtoul(argv[count], NULL, 0);
		switch(devNo){
		case 1:
			mms101_data[0].channelEnable = 1;
			break;
		case 2:
			mms101_data[1].channelEnable = 1;
			break;
		case 3:
			mms101_data[2].channelEnable = 1;
			break;
		case 4:
			mms101_data[3].channelEnable = 1;
			break;
		case 5:
			mms101_data[4].channelEnable = 1;
			break;
		default:
			mms101_usage();
			return(1);
		}
	}
	// set measurement number
	measNumMax = (int)strtoul(argv[argc-1], NULL, 0);
	if(measNumMax >= MEAS_NUM_MAX){
		mms101_usage();
		return(1);
	}
	
	temperatureCorrectionRequest = 0x00;
	measureStopRequest = 0x00;
	measureEndFlag = 0;
	
	// ----------------------------------------
	// setup SPI driver
	// ----------------------------------------
	if(spi_open(&spi_dev) != 0){
		return(1);
	}
	// ----------------------------------------
	// RESET
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			if(mms101_spi1_reset(devNo) != 0){
				return(1);
			}
		}
	}
	// ----------------------------------------
	// wait for STANDBY State
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			retryCount = 0;
			do{
				usleep(20*1000);						// wait 20msec
				if(mms101_spi1_status(devNo) != 0){
					return(1);
				}
				retryCount++;
				if(retryCount>5){
					printf("RESET Error\n");
					return(1);
				}
			} while(mms101_data[devNo].stateID != (uint8_t)MMS101_SPI1_STT_STANDBY);
		}
	}
	// ----------------------------------------
	// BOOT
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			if(mms101_spi1_boot(devNo) != 0){
				return(1);
			}
		}
	}
	// ----------------------------------------
	// wait for BOOT operation to complete
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			retryCount = 0;
			do{
				usleep(20*1000);						// wait 20msec
				if(mms101_spi1_status(devNo) != 0){
					return(1);
				}
				retryCount++;
				if(retryCount>5){
					printf("BOOT Error\n");
					return(1);
				}
			} while(mms101_data[devNo].stateID != (uint8_t)MMS101_SPI1_STT_READY);
		}
	}
	// ----------------------------------------
	// COEFF Fx,Fy,Fz,Mx,My,Mz
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			if(mms101_spi1_coeff(devNo) != 0){
				return(1);
			}
		}
	}
	// ----------------------------------------
	// INTERVAL(update temperature correction)
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			if(mms101_spi1_interval(devNo) != 0){
				return(1);
			}
		}
	}
	// ----------------------------------------
	// START
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			if(mms101_spi1_start(devNo) != 0){
				return(1);
			}
		}
	}
	// ----------------------------------------
	// Initialize Timer & Counter
	// ----------------------------------------
	gettimeofday(&startTime, NULL);
	preGetTime.tv_sec = startTime.tv_sec;
	preGetTime.tv_usec = startTime.tv_usec;
	waitTime = (int)WAIT_TIME_FOR_STARTUP;
	count = 0;
	// ----------------------------------------
	// Measure
	// ----------------------------------------
	printf("devNo,time[s],Fx,Fy,Fz,Mx,My,Mz\n");
	do{
		//--------------------------------------------------------------------------
		// Waiting for correction conversion completion
		//--------------------------------------------------------------------------
		do{
			usleep(1);
			elapsedTime = getElapsedTimeUs(&preGetTime, &tv);
		}while(elapsedTime < waitTime);
		preGetTime.tv_sec = tv.tv_sec;
		preGetTime.tv_usec = tv.tv_usec;
		waitTime = (int)DATA_RATE_TIME;
		// ----------------------------------------
		// DATA2
		// ----------------------------------------
		for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
			if(mms101_data[devNo].channelEnable == 1){
				if(mms101_spi1_data2(devNo) != 0){
					return(1);
				}
			}
		}
		// ----------------------------------------
		// Matrix operatoin
		// ----------------------------------------
		for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
			if(mms101_data[devNo].channelEnable == 1){
					mms101_calc_matrix(devNo);
			}
		}
		// ----------------------------------------
		// if temperature correction is required
		// ----------------------------------------
		if(temperatureCorrectionRequest==0x01){
			temperatureCorrectionRequest = 0x00;
			for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
				if(mms101_data[devNo].channelEnable == 1){
					if(mms101_spi1_restart(devNo) != 0){
						return(1);
					}
				}
			}
			waitTime = (int)WAIT_TIME_FOR_TEMPERATURE;
		}
		// ----------------------------------------
		// measure end ?
		// ----------------------------------------
		count++;
		if(count >= measNumMax){				// measurement times
			measureEndFlag = 0x01;	
		}
		if(measureStopRequest == 0x01){			// measurement stop request
			measureStopRequest = 0x00;
			measureEndFlag = 0x01;
		}
	}while(measureEndFlag == 0x00);
	// ----------------------------------------
	// STOP
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			if(mms101_spi1_stop(devNo) != 0){
				return(1);
			}
		}
	}
	// ----------------------------------------
	// wait for READY state
	// ----------------------------------------
	for(devNo=0; devNo<MMS101_DEVICE_NUM; devNo++){
		if(mms101_data[devNo].channelEnable == 1){
			retryCount = 0;
			do{
				usleep(20*1000);						// wait 20msec
				if(mms101_spi1_status(devNo) != 0){
					return(1);
				}
				retryCount++;
				if(retryCount>5){
					printf("STOP Error\n");
					return(1);
				}
			} while(mms101_data[devNo].stateID != (uint8_t)MMS101_SPI1_STT_READY);
		}
	}
	// ----------------------------------------
	// Close SPI
	// ----------------------------------------
	spi_close(&spi_dev);

	return(0);
}

/*!
//==============================================================================
	@brief		START
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_start(uint8_t devNo)
{
#ifdef DEBUG
	uint8_t count;
#endif
	//------------------
	// START Command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_START;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 1) < 0){
		printf("CN%d, ERROR: START command error\n", devNo+1);
		return(1); 	// error
	}

#ifdef DEBUG
	printf("START Command:");
	for(count=0;count<1;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif

	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: START command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	return(0);
}

/*!
//==============================================================================
	@brief		DATA2
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_data2(uint8_t devNo)
{
	uint32_t dataWork;
	uint8_t  axisNo;
#ifdef DEBUG
	uint8_t  count;
#endif
	//------------------
	// DATA2 command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_DATA2;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 21) < 0){
		printf("CN%d, ERROR: DATA2 command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("DATA2 Command:");
	for(count=0;count<21;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: DATA2 command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode     = rx_buf[0];
	mms101_data[devNo].measureStatus  = (uint16_t)rx_buf[1] << 8;
	mms101_data[devNo].measureStatus += (uint16_t)rx_buf[2];
	for(axisNo=0; axisNo<6; axisNo++){
		dataWork  = ((uint32_t)rx_buf[3 + axisNo*3]) << 16;
		dataWork += ((uint32_t)rx_buf[4 + axisNo*3]) << 8;
		dataWork +=  (uint32_t)rx_buf[5 + axisNo*3];
		if(dataWork >= 0x00800000){
			dataWork |= 0xff000000;
		}
		mms101_data[devNo].adc[axisNo] = (int32_t)dataWork;
	}
	return (0);
}

/*!
//==============================================================================
	@brief		RESTART
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_restart(uint8_t devNo)
{
#ifdef DEBUG
	uint8_t  count;
#endif
	//------------------
	// RESTART command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_RESTART;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 1) < 0){
		printf("CN%d, ERROR: RESTART command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("RESTART Command:");
	for(count=0;count<1;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: RESTART command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	return(0);
}

/*!
//==============================================================================
	@brief		BOOT
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_boot(uint8_t devNo)
{
#ifdef DEBUG
	uint8_t  count;
#endif
	//------------------
	// BOOT Command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_BOOT;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 1) < 0){
		printf("CN%d, ERROR: BOOT command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("BOOT Command:");
	for(count=0;count<1;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: BOOT command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	return(0);
}

/*!
//==============================================================================
	@brief		STOP
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_stop(uint8_t devNo)
{
#ifdef DEBUG
	uint8_t count;
#endif
	//------------------
	// STOP Command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_STOP;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 1) < 0){
		printf("CN%d, ERROR: STOP command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("STOP Command:");
	for(count=0;count<1;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: STOP command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	return(0);
}

/*!
//==============================================================================
	@brief		RESET
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_reset(uint8_t devNo)
{
#ifdef DEBUG
	uint8_t  count;
#endif
	//------------------
	// RESET Command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_RESET;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 1) < 0){
		printf("CN%d, ERROR: RESET command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("RESET Command:");
	for(count=0;count<1;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: RESET command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	return(0);
}


/*!
//==============================================================================
	@brief		STATUS
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_status(uint8_t devNo)
{
#ifdef DEBUG
	uint8_t  count;
#endif
	//------------------
	// STATUS Command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_STATUS;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 4) < 0){
		printf("CN%d, ERROR: STATUS command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("STATUS Command:");
	for(count=0;count<4;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: STATUS command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode     = rx_buf[0];
	mms101_data[devNo].measureStatus  = (uint16_t)rx_buf[1] << 8;
	mms101_data[devNo].measureStatus += (uint16_t)rx_buf[2];
	mms101_data[devNo].stateID        = rx_buf[3];
	return (0);
}

/*!
//==============================================================================
	@brief		VERSION
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_version(uint8_t devNo)
{
#ifdef DEBUG
	uint8_t  count;
#endif
	//------------------
	// VERSION Command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_VERSION;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 7) < 0){
		printf("CN%d, ERROR: STATUS command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("VERSION Command:");
	for(count=0;count<7;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: STATUS command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	mms101_data[devNo].hwVer[0]   = rx_buf[1];
	mms101_data[devNo].hwVer[1]   = rx_buf[2];
	mms101_data[devNo].fwVer[0]   = rx_buf[3];
	mms101_data[devNo].fwVer[1]   = rx_buf[4];
	mms101_data[devNo].fwVer[2]   = rx_buf[5];
	mms101_data[devNo].fwVer[3]   = rx_buf[6];
	return (0);
}


/*!
//==============================================================================
	@brief		COEFF
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_coeff(uint8_t devNo)
{
	uint32_t dataWork;
	uint8_t  axisNo;
#ifdef DEBUG
	uint8_t  count;
#endif	
	//------------------
	// COEFF_FX command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_COEFF_FX;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 19) < 0){
		printf("CN%d, ERROR: COEFF_FX command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("COEFF_FX Command:");
	for(count=0;count<19;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: COEFF_FX command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	for(axisNo=0; axisNo<6; axisNo++){
		dataWork  = ((uint32_t)rx_buf[1 + axisNo*3]) << 16;
		dataWork += ((uint32_t)rx_buf[2 + axisNo*3]) << 8;
		dataWork +=  (uint32_t)rx_buf[3 + axisNo*3];
		if(dataWork >= 0x00800000){
			dataWork |= 0xff000000;
		}
		mms101_data[devNo].coeff[0][axisNo] = (int32_t)dataWork;
	}

	//------------------
	// COEFF_FY command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_COEFF_FY;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 19) < 0){
		printf("CN%d, ERROR: COEFF_FY command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("COEFF_FY Command:");
	for(count=0;count<19;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: COEFF_FY command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode   = rx_buf[0];
	for(axisNo=0; axisNo<6; axisNo++){
		dataWork  = ((uint32_t)rx_buf[1 + axisNo*3]) << 16;
		dataWork += ((uint32_t)rx_buf[2 + axisNo*3]) << 8;
		dataWork +=  (uint32_t)rx_buf[3 + axisNo*3];
		if(dataWork >= 0x00800000){
			dataWork |= 0xff000000;
		}
		mms101_data[devNo].coeff[1][axisNo] = (int32_t)dataWork;
	}

	//------------------
	// COEFF_FZ command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_COEFF_FZ;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 19) < 0){
		printf("CN%d, ERROR: COEFF_FZ command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("COEFF_FZ Command:");
	for(count=0;count<19;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: COEFF_FZ command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	for(axisNo=0; axisNo<6; axisNo++){
		dataWork  = ((uint32_t)rx_buf[1 + axisNo*3]) << 16;
		dataWork += ((uint32_t)rx_buf[2 + axisNo*3]) << 8;
		dataWork +=  (uint32_t)rx_buf[3 + axisNo*3];
		if(dataWork >= 0x00800000){
			dataWork |= 0xff000000;
		}
		mms101_data[devNo].coeff[2][axisNo] = (int32_t)dataWork;
	}
	
	//------------------
	// COEFF_MX command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_COEFF_MX;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 19) < 0){
		printf("CN%d, ERROR: COEFF_MX command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("COEFF_MX Command:");
	for(count=0;count<19;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: COEFF_MX command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode   = rx_buf[0];
	for(axisNo=0; axisNo<6; axisNo++){
		dataWork  = ((uint32_t)rx_buf[1 + axisNo*3]) << 16;
		dataWork += ((uint32_t)rx_buf[2 + axisNo*3]) << 8;
		dataWork +=  (uint32_t)rx_buf[3 + axisNo*3];
		if(dataWork >= 0x00800000){
			dataWork |= 0xff000000;
		}
		mms101_data[devNo].coeff[3][axisNo] = (int32_t)dataWork;
	}

	//------------------
	// COEFF_MY command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_COEFF_MY;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 19) < 0){
		printf("CN%d, ERROR: COEFF_MY command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("COEFF_MY Command:");
	for(count=0;count<19;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: COEFF_MY command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	for(axisNo=0; axisNo<6; axisNo++){
		dataWork  = ((uint32_t)rx_buf[1 + axisNo*3]) << 16;
		dataWork += ((uint32_t)rx_buf[2 + axisNo*3]) << 8;
		dataWork +=  (uint32_t)rx_buf[3 + axisNo*3];
		if(dataWork >= 0x00800000){
			dataWork |= 0xff000000;
		}
		mms101_data[devNo].coeff[4][axisNo] = (int32_t)dataWork;
	}
	
	//------------------
	// COEFF_MZ command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_COEFF_MZ;
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 1, rx_buf, 19) < 0){
		printf("CN%d, ERROR: COEFF_MZ command error\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("COEFF_MZ Command:");
	for(count=0;count<19;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: COEFF_MZ command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	for(axisNo=0; axisNo<6; axisNo++){
		dataWork  = ((uint32_t)rx_buf[1 + axisNo*3]) << 16;
		dataWork += ((uint32_t)rx_buf[2 + axisNo*3]) << 8;
		dataWork +=  (uint32_t)rx_buf[3 + axisNo*3];
		if(dataWork >= 0x00800000){
			dataWork |= 0xff000000;
		}
		mms101_data[devNo].coeff[5][axisNo] = (int32_t)dataWork;
	}

	return (0);
}

/*!
//==============================================================================
	@brief		INTERVAL
	@param		uint8_t devNo
	@return		int (0:OK/1:NG)
//==============================================================================
*/
int mms101_spi1_interval(uint8_t devNo)
{
#ifdef DEBUG
	uint8_t  count;
#endif
	//------------------
	// INTERVAL command
	//------------------
	tx_buf[0] = MMS101_SPI1_CMD_INTERVAL;
	tx_buf[1] = (uint8_t)((uint32_t)INTERVAL_NUM >> 16);
	tx_buf[2] = (uint8_t)((uint32_t)INTERVAL_NUM >> 8);
	tx_buf[3] = (uint8_t)((uint32_t)INTERVAL_NUM >> 0);
	if(spi_send_receive_data(devNo, &spi_dev, tx_buf, 4, rx_buf, 1) < 0){
		printf("CN%d, ERROR: INTERVAL command\n", devNo+1);
		return(1); 	// error
	}
#ifdef DEBUG
	printf("INTERVAL Command:");
	for(count=0;count<1;count++){
		printf("%d,",rx_buf[count]);
	}
	printf("\n");
#endif
	if(rx_buf[0] != 0x00){
		printf("CN%d, ERROR: INTERVAL command status error\n", devNo+1);
		return(1); 	// error
	}
	mms101_data[devNo].statusCode = rx_buf[0];
	return(0);
}


/*!
//==============================================================================
	@brief		Matrix calculation
	@param		uint8_t devNo
	@return		void
//==============================================================================
*/
void mms101_calc_matrix(uint8_t devNo)
{
	int64_t temp64;
	uint8_t axisNo;

	for(axisNo=0; axisNo<6; axisNo++)
	{
		temp64  = (int64_t)mms101_data[devNo].coeff[axisNo][0] * (int64_t)mms101_data[devNo].adc[0];
		temp64 += (int64_t)mms101_data[devNo].coeff[axisNo][1] * (int64_t)mms101_data[devNo].adc[1];
		temp64 += (int64_t)mms101_data[devNo].coeff[axisNo][2] * (int64_t)mms101_data[devNo].adc[2];
		temp64 += (int64_t)mms101_data[devNo].coeff[axisNo][3] * (int64_t)mms101_data[devNo].adc[3];
		temp64 += (int64_t)mms101_data[devNo].coeff[axisNo][4] * (int64_t)mms101_data[devNo].adc[4];
		temp64 += (int64_t)mms101_data[devNo].coeff[axisNo][5] * (int64_t)mms101_data[devNo].adc[5];
		temp64 = temp64 / (int64_t)RIGHT_BIT_SHIFT_11;

		if(temp64 < (int64_t)0xFFFFFFFFFF800000){
			mms101_data[devNo].matrix[axisNo] = (int32_t)0xFF800000;		// minimum
		}else if(temp64 > (int64_t)0x00000000007FFFFF){
			mms101_data[devNo].matrix[axisNo] = (int32_t)0x007FFFFF;		// maximum
		}else{	
			mms101_data[devNo].matrix[axisNo] = (int32_t)temp64;
		}
	}
	
	// Output matrix data
	printf("CN%d", devNo+1);
	printf(",%f"  , getElapsedTime(&startTime, 0));
	printf(",%.3f", (double)mms101_data[devNo].matrix[0] / 1000);
	printf(",%.3f", (double)mms101_data[devNo].matrix[1] / 1000);
	printf(",%.3f", (double)mms101_data[devNo].matrix[2] / 1000);
	printf(",%.5f", (double)mms101_data[devNo].matrix[3] / 100000);
	printf(",%.5f", (double)mms101_data[devNo].matrix[4] / 100000);
	printf(",%.5f", (double)mms101_data[devNo].matrix[5] / 100000);
	printf("\n");

}


/*!
//==============================================================================
	@brief		Usage

	@param		void
	@return		void
	
//==============================================================================
	@par		Operations Description:

//==============================================================================
*/
void mms101_usage(void)
{
	printf("\n");
	printf("   Usage: mms101 [channel(1~5)] [measurement number]\n");
	printf("\n");
}

