//******************************************************************************
/*!
	@file		spidrv.c
	@brief		SPI Module Function Source File

				copyright: MITSUMI Electric Co.,LTD
	@attention	none
	@warning	MITSUMI CONFIDENTIAL
*/
//******************************************************************************
/*******************************************************************************
* History : DD.MM.YYYY Version  Description
*         : 24.12.2024 1.0.0.0  First Release
*******************************************************************************/
#define __SPIDRV_C__

//==============================================================================
//{### MODULES USED
//==============================================================================
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <linux/spi/spidev.h>
#include <sys/time.h>
#include "spidrv.h"
#include "mms101.h"

//==============================================================================
//{### FUNCTIONS
//==============================================================================
/*!
//==============================================================================
	@brief		SPI open
	@param		spi_device_t *spi_device
	@return		int
//==============================================================================
*/
int spi_open(spi_device_t *spi_device)
{
	int status = 0;
	int fd = 0;
	char devName[50];
	uint8_t csbNo;
	
	memset(spi_device, 0, sizeof(spi_device_t));

	//----------------------------------------
	// set CSB port
	//----------------------------------------
	for(csbNo=0; csbNo<SPI_CSB_MAX; csbNo++){

		// enable GPIO<n>
		// "/sys/class/gpio/export" <- "<n>"
		strncpy(devName, "/sys/class/gpio/export", 40);
		printf("%s\n",devName);
		fd = open(devName, O_WRONLY);
		if(fd < 0){
			printf("ERROR: GPIO%d export open error.\n", csbNo);
			status = -1;
		}
		write(fd, csGpioNo[csbNo], 2);
		close(fd);

		// wait time until GPIO<n> is generated : 0.5sec
		usleep(500 * 1000);

		// set GPIO<n> as output
		// "/sys/class/gpio/gpio<n>/direction" <- "out"
		strncpy(devName, "/sys/class/gpio/gpio", 40);
		strncat(devName, csGpioNo[csbNo], 40);
		strncat(devName, "/direction", 40);
		printf("%s\n",devName);
		fd = open(devName, O_WRONLY);
		if(fd < 0){
			printf("ERROR: GPIO%d direction open error.\n", csbNo);
			status = -1;
		}
		write(fd, "out", 4);
		close(fd);
		
		// set GPIO<n> value
		// "/sys/class/gpio/gpio<n>/value" <- "0" or "1"
		strncpy(devName, "/sys/class/gpio/gpio", 40);
		strncat(devName, csGpioNo[csbNo], 40);
		strncat(devName, "/value", 40);
		printf("%s\n",devName);
		fd = open(devName, O_WRONLY);
		if(fd < 0){
			printf("ERROR: GPIO%d value open error.\n", csbNo);
			status = -1;
		}
		// store fd for CSB<n>
		spi_device->fd_csb[csbNo] = fd;

		// set CSB=high
		if(spi_disable(csbNo, spi_device) < 0){
			status = -1;
		}
	}

	//----------------------------------------
	// set SPI Module
	//----------------------------------------
	strncpy(devName, "/dev/spidev0.0", 40);
	fd = open(devName, O_RDWR);
	if(fd < 0){
		printf("Error: spi device open error\n");
		status = -1;
	}
	// store fd for SPI module
	spi_device->fd = fd;
	
	spi_device->mode				= SPI_MODE;
	spi_device->tr[0].speed_hz		= SPI_SPEED_HZ;
	spi_device->tr[0].delay_usecs	= SPI_DELAY_USECS;
	spi_device->tr[0].bits_per_word	= SPI_SPI_BITS;
	spi_device->tr[0].cs_change		= SPI_CS_CHANGE;

	if(ioctl(spi_device->fd, SPI_IOC_WR_MODE, &(spi_device->mode)) < 0){
		printf("ERROR: set spi mode(wr)\n");
		status = -1;
	}

	if(ioctl(spi_device->fd, SPI_IOC_RD_MODE, &(spi_device->mode)) < 0){
		printf("ERROR: set spi mode(rd)\n");
		status = -1;
	}

	if(ioctl(spi_device->fd, SPI_IOC_WR_MAX_SPEED_HZ, &(spi_device->tr[0].speed_hz)) < 0){
		printf("ERROR: set spi speed(wr)\n");
		status = -1;
	}

	if(ioctl(spi_device->fd, SPI_IOC_RD_MAX_SPEED_HZ, &(spi_device->tr[0].speed_hz)) < 0){
		printf("ERROR: set spi speed(rd)\n");
		status = -1;
	}

	printf("Send dummy clock\n");
	//----------------------------------------
	// send dummy clock
	// The initial value of SCLK is low.
	// Until the clock is sent, SCLK will not become high.
	// Therefore, insert a dummy clock to set SCLK to high.
	//----------------------------------------
	tx_buf[0] = 0x00;
	spi_device->tr[0].tx_buf = (ulong)tx_buf;
	spi_device->tr[0].rx_buf = 0x00;
	spi_device->tr[0].len = 1;
	if(ioctl(spi_device->fd, SPI_IOC_MESSAGE(1), spi_device->tr) < 0){
		printf("ERROR: send dummy clock\n");
		status = -1;
	}
	
	return status;
}


/*!
//==============================================================================
	@brief		SPI Close
	@param		spi_device_t *spi_device
	@return		int
//==============================================================================
*/
int spi_close(spi_device_t *spi_device)
{
	int status = 0;
	uint8_t csbNo;
	//----------------------------------------
	// release CSB port
	//----------------------------------------
	for(csbNo=0; csbNo<SPI_CSB_MAX; csbNo++){
		if(close(spi_device->fd_csb[csbNo]) < 0){
			status = -1;
		}
	}
	//----------------------------------------
	// release SPI Module
	//----------------------------------------
	if(close(spi_device->fd) < 0){
		status = -1;
	}
	return status;
}

/*!
//==============================================================================
	@brief		SPI Enable Module
	@param		uint8_t csbNo
	@param		spi_device_t *spi_device
	@return		int
//==============================================================================
*/
int spi_enable(uint8_t csbNo, spi_device_t *spi_device)
{
	//----------------------------------------
	// CSB=Low
	//----------------------------------------
	return write(spi_device->fd_csb[csbNo], "0", 2);
}

/*!
//==============================================================================
	@brief		SPI Disable Module
	@param		uint8_t csbNo
	@param		spi_device_t *spi_device
	@return		int
//==============================================================================
*/
int spi_disable(uint8_t csbNo, spi_device_t *spi_device)
{
	//----------------------------------------
	// CSB=High
	//----------------------------------------
	return write(spi_device->fd_csb[csbNo], "1", 2);
}


/*!
//==============================================================================
	@brief		SPI Send Data Module
	@param		uint8_t csbNo             : csb number
	@param		spi_device_t *spi_device  : spi device infomation
	@param		uint8_t *txbuf            : transmitting data buffer
	@param		uint8_t txlen             : number of transmitting data
	@return		int
//==============================================================================
*/
int spi_send_data(uint8_t csbNo, spi_device_t *spi_device, uint8_t *txbuf, uint8_t txlen)
{
	int status = 0;

	// set CSB=Low
	if(spi_enable(csbNo, spi_device) < 0){
		status = -1;
	}
	// send data
	spi_device->tr[0].tx_buf = (ulong)txbuf;
	spi_device->tr[0].rx_buf = 0;
	spi_device->tr[0].len = txlen;
	if(ioctl(spi_device->fd, SPI_IOC_MESSAGE(1), spi_device->tr) < 0){
		status = -1;
	}
	// set CSB=High
	if(spi_disable(csbNo, spi_device) < 0){
		status = -1;
	}
	return status;
}

/*!
//==============================================================================
	@brief		SPI Receive Data Module
	@param		uint8_t csbNo             : csb number
	@param		spi_device_t *spi_device  : spi device infomation
	@param		uint8_t *rxbuf            : received data buffer
	@param		uint8_t rxlen             : number of received data
	@return		int
//==============================================================================
*/
int spi_receive_data(uint8_t csbNo, spi_device_t *spi_device, uint8_t *rxbuf, uint8_t rxlen)
{
	int status = 0;

	// set CSB=Low
	if(spi_enable(csbNo, spi_device) < 0){
		status = -1;
	}
	// receive data
	spi_device->tr[0].tx_buf = 0;
	spi_device->tr[0].rx_buf = (ulong)rxbuf;
	spi_device->tr[0].len = rxlen;
	if(ioctl(spi_device->fd, SPI_IOC_MESSAGE(1), spi_device->tr) < 0){
		status = -1;
	}
	// set CSB=High
	if(spi_disable(csbNo, spi_device) < 0){
		status = -1;
	}
	return status;
}

/*!
//==============================================================================
	@brief		SPI Send Receive Data Module
	@param		uint8_t csbNo             : number of device
	@param		spi_device_t *spi_device  : spi device infomation
	@param		uint8_t *txbuf            : transmitting data buffer
	@param		uint8_t txlen             : number of transmitting data
	@param		uint8_t *rxbuf            : received data buffer
	@param		uint8_t rxlen             : number of received data
	@return		int
//==============================================================================
*/
int spi_send_receive_data(uint8_t csbNo, spi_device_t *spi_device, uint8_t *txbuf, uint8_t txlen, uint8_t *rxbuf, uint8_t rxlen)
{
	int status = 0;

	// set CSB=Low
	if(spi_enable(csbNo, spi_device) < 0){
		status = -1;
	}
	// send data
	spi_device->tr[0].tx_buf = (ulong)txbuf;
	spi_device->tr[0].rx_buf = 0;
	spi_device->tr[0].len = txlen;
	if(ioctl(spi_device->fd, SPI_IOC_MESSAGE(1), spi_device->tr) < 0){
		status = -1;
	}
	// receive data
	spi_device->tr[0].tx_buf = 0;
	spi_device->tr[0].rx_buf = (ulong)rxbuf;
	spi_device->tr[0].len = rxlen;
	if(ioctl(spi_device->fd, SPI_IOC_MESSAGE(1), spi_device->tr) < 0){
		status = -1;
	}
	// set CSB=High
	if(spi_disable(csbNo, spi_device) < 0){
		status = -1;
	}
	return status;
}
	
	
/*!
//==============================================================================
	@brief 		Get Elapsed Time(usec)
	@param		struct timeval *tvStart : Base Time
	@param		struct timeval *tvUpdate : Copy Now Time (Null:Not copy)
	@return 	int : elapsed time(usec) (<= 2147483647usec)
//==============================================================================
*/
int getElapsedTimeUs(struct timeval *tvStart, struct timeval *tvUpdate){
	struct timeval tvNow;
	int elapsed;

	gettimeofday(&tvNow, NULL);

	elapsed = (tvNow.tv_sec - tvStart->tv_sec) * 1000000;
	elapsed += (tvNow.tv_usec - tvStart->tv_usec);

	if(tvUpdate != NULL){
		tvUpdate->tv_sec = tvNow.tv_sec;
		tvUpdate->tv_usec = tvNow.tv_usec;
	}

	return elapsed;
}

/*!
//==============================================================================
	@brief 		Read Elapsed Time(sec)
	@param		struct timeval *tvStart : Base Time
	@param		struct timeval *tvUpdate : Copy Now Time (Null:Not copy)
	@return 	double : elapsed time(sec)
//==============================================================================
*/
double getElapsedTime(struct timeval *tvStart, struct timeval *tvUpdate){
	struct timeval tvNow;
	double elapsed;

	gettimeofday(&tvNow, NULL);

	elapsed = (double)tvNow.tv_sec + ((double)tvNow.tv_usec) / 1000000;
	elapsed -= ((double)tvStart->tv_sec + ((double)tvStart->tv_usec) / 1000000);

	if(tvUpdate != NULL){
		tvUpdate->tv_sec = tvNow.tv_sec;
		tvUpdate->tv_usec = tvNow.tv_usec;
	}

	return elapsed;
}

